<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;

class UserController extends Controller
{
    // =========================
    // 🔐 Registration
    // =========================
    public function register(Request $request)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'email' => 'required|email|unique:users,email',
                'password' => 'required|min:6'
            ]);

            $user = User::create([
                'name' => $validated['name'],
                'email' => $validated['email'],
                'password' => Hash::make($validated['password'])
            ]);

            $token = $user->createToken('api-token')->plainTextToken;

            return response()->json([
                'status' => true,
                'message' => 'User registered successfully',
                'token' => $token,
                'user' => $user
            ], 201);
        } catch (\Exception $e) {
            Log::error('Registration Error', [
                'message' => $e->getMessage()
            ]);

            return response()->json([
                'status' => false,
                'message' => 'Something went wrong'
            ], 500);
        }
    }

    // =========================
    // 🔓 Login
    // =========================
    // public function login(Request $request)
    // {
    //     try {
    //         $validated = $request->validate([
    //             'email' => 'required|email',
    //             'password' => 'required'
    //         ]);

    //         if (!Auth::attempt($validated)) {
    //             return response()->json([
    //                 'status' => false,
    //                 'message' => 'Invalid credentials'
    //             ], 401);
    //         }

    //         $user = Auth::user();
    //         $token = $user->createToken('api-token')->plainTextToken;

    //         return response()->json([
    //             'status' => true,
    //             'message' => 'Login successful',
    //             'token' => $token,
    //             'user' => $user
    //         ], 200);
    //     } catch (\Illuminate\Validation\ValidationException $e) {
    //         Log::warning('Login Validation Error', [
    //             'errors' => $e->errors()
    //         ]);

    //         return response()->json([
    //             'status' => false,
    //             'errors' => $e->errors()
    //         ], 422);
    //     } catch (\Exception $e) {
    //         Log::error('Login Error', [
    //             'message' => $e->getMessage()
    //         ]);

    //         return response()->json([
    //             'status' => false,
    //             'message' => 'Something went wrong'
    //         ], 500);
    //     }
    // }


    public function login(Request $request)
    {
        $credentials = $request->validate([
            'email' => 'required|email',
            'password' => 'required'
        ]);

        if (!Auth::attempt($credentials)) {
            return back()->withErrors([
                'email' => 'Invalid credentials'
            ]);
        }

        $request->session()->regenerate();

        return redirect()->intended('/');
    }

    public function logout(Request $request)
    {
        Auth::logout(); // User logout

        $request->session()->invalidate(); // Destroy session
        $request->session()->regenerateToken(); // CSRF regenerate

        return redirect('/login'); // Redirect to login page
    }
}
