import { ComponentType } from 'react';
import { AxisInterval, AxisTick, RenderableAxisProps, PresentationAttributesAdaptChildEvent, AxisDomain, ScaleType, AxisDomainTypeInput, EvaluatedAxisDomainType, TickProp, XAxisTickContentProps } from '../util/types';
import { XAxisOrientation, XAxisPadding } from '../state/cartesianAxisSlice';
import { CustomScaleDefinition } from '../util/scale/CustomScaleDefinition';
interface XAxisProps extends Omit<RenderableAxisProps, 'domain' | 'scale' | 'tick'> {
    /**
     * The type of axis.
     *
     * `category`: Treats data as distinct values.
     * Each value is in the same distance from its neighbors, regardless of their actual numeric difference.
     *
     * `number`: Treats data as continuous range.
     * Values that are numerically closer are placed closer together on the axis.
     *
     * `auto`: the type is inferred based on the chart layout.
     *
     * @defaultValue category
     */
    type?: AxisDomainTypeInput;
    /**
     * Unique ID that represents this XAxis.
     * Required when there are multiple XAxes.
     *
     * @defaultValue 0
     */
    xAxisId?: string | number;
    /**
     * Height of the axis in pixels.
     *
     * @defaultValue 30
     */
    height?: number;
    /**
     * If set true, flips ticks around the axis line, displaying the labels inside the chart instead of outside.
     * @defaultValue false
     */
    mirror?: boolean;
    /**
     * The orientation of axis
     * @defaultValue bottom
     */
    orientation?: XAxisOrientation;
    /**
     * Specify the domain of axis when the axis is a number axis.
     *
     * If undefined, then the domain is calculated based on the data and dataKeys.
     *
     * The length of domain should be 2, and we will validate the values in domain.
     *
     * Each element in the array can be a number, 'auto', 'dataMin', 'dataMax', a string like 'dataMin - 20', 'dataMax + 100',
     * or a function that accepts a single argument and returns a number.
     *
     * If any element of domain is set to be 'auto', comprehensible scale ticks will be calculated, and the final domain of axis is generated by the ticks.
     * If a function, receives '[dataMin, dataMax]', and must return a computed domain as '[min, max]'.
     *
     * @example <XAxis type="number" domain={['dataMin', 'dataMax']} />
     * @example <XAxis type="number" domain={[0, 'dataMax']} />
     * @example <XAxis type="number" domain={['auto', 'auto']} />
     * @example <XAxis type="number" domain={[0, 'dataMax + 1000']} />
     * @example <XAxis type="number" domain={['dataMin - 100', 'dataMax + 100']} />
     * @example <XAxis type="number" domain={[dataMin => (0 - Math.abs(dataMin)), dataMax => (dataMax * 2)]} />
     * @example <XAxis type="number" domain={([dataMin, dataMax]) => { const absMax = Math.max(Math.abs(dataMin), Math.abs(dataMax)); return [-absMax, absMax]; }} />
     * @example <XAxis type="number" domain={[0, 100]} allowDataOverflow />
     */
    domain?: AxisDomain;
    /**
     * Scale function determines how data values are mapped to visual values.
     * In other words, decided the mapping between data domain and coordinate range.
     *
     * If undefined, or 'auto', the scale function is created internally according to the type of axis and data.
     *
     * You can define a custom scale, either as a string shortcut to a d3 scale, or as a complete scale definition object.
     *
     * @defaultValue auto
     * @example <XAxis scale="log" />
     * @example
     * import { scaleLog } from 'd3-scale';
     * const scale = scaleLog().base(Math.E);
     * <XAxis scale={scale} />
     */
    scale?: ScaleType | CustomScaleDefinition | CustomScaleDefinition<string> | CustomScaleDefinition<number> | CustomScaleDefinition<Date>;
    /**
     * Defines how the individual label text is rendered.
     * This controls the settings for individual ticks; on a typical axis, there are multiple ticks, depending on your data.
     *
     * If you want to customize the overall axis label, use the `label` prop instead.
     *
     * Options:
     * - `false`: Do not render any tick labels.
     * - `true`: Render tick labels with default settings.
     * - `object`: An object of props to be merged into the internally calculated tick props.
     * - `ReactElement`: A custom React element to be used as the tick label.
     * - `function`: A function that returns a React element for custom rendering of tick labels.
     *
     * @defaultValue true
     */
    tick?: TickProp<XAxisTickContentProps>;
    /**
     * Ticks can be any type when the axis is the type of category
     * Ticks must be numbers when the axis is the type of number
     */
    ticks?: ReadonlyArray<AxisTick>;
    /**
     * Axis padding is the distance between the edge of plot area and the first/last tick.
     *
     * @defaultValue {"left":0,"right":0}
     */
    padding?: XAxisPadding;
    /**
     * The minimum gap between two adjacent tick labels
     *
     * @defaultValue 5
     */
    minTickGap?: number;
    /**
     * If set 0, all the ticks will be shown. If set preserveStart", "preserveEnd" or "preserveStartEnd", the ticks which is to be shown or hidden will be calculated automatically.
     * @defaultValue preserveEnd
     */
    interval?: AxisInterval;
    /**
     * The margin between tick line and tick.
     */
    tickMargin?: number;
}
export type Props = Omit<PresentationAttributesAdaptChildEvent<any, SVGElement>, 'scale' | 'ref'> & XAxisProps;
export declare const xAxisDefaultProps: {
    readonly allowDataOverflow: boolean;
    readonly allowDecimals: boolean;
    readonly allowDuplicatedCategory: boolean;
    readonly angle: number;
    readonly axisLine: true;
    readonly height: number;
    readonly hide: false;
    readonly includeHidden: boolean;
    readonly interval: AxisInterval;
    readonly label: false;
    readonly minTickGap: number;
    readonly mirror: boolean;
    readonly orientation: XAxisOrientation;
    readonly padding: XAxisPadding;
    readonly reversed: boolean;
    readonly scale: ScaleType | CustomScaleDefinition<import("../util/types").CategoricalDomainItem> | CustomScaleDefinition<string> | CustomScaleDefinition<number> | CustomScaleDefinition<Date>;
    readonly tick: TickProp<any>;
    readonly tickCount: number | undefined;
    readonly tickLine: true;
    readonly tickSize: 6;
    readonly type: EvaluatedAxisDomainType;
    readonly xAxisId: 0;
};
/**
 * @consumes CartesianViewBoxContext
 * @provides CartesianLabelContext
 */
export declare const XAxis: ComponentType<Props>;
export {};
