import { ComponentType, SVGProps } from 'react';
import { AxisDomainTypeInput, AxisInterval, AxisTick, RenderableAxisProps, PresentationAttributesAdaptChildEvent, AxisDomain, ScaleType, EvaluatedAxisDomainType, TickProp, YAxisTickContentProps } from '../util/types';
import { YAxisOrientation, YAxisPadding, YAxisWidth } from '../state/cartesianAxisSlice';
import { CustomScaleDefinition } from '../util/scale/CustomScaleDefinition';
interface YAxisProps extends Omit<RenderableAxisProps, 'axisLine' | 'domain' | 'scale' | 'tick'> {
    /**
     * Determines how the axis line is drawn. Options:
     * - `true`: the axis line is drawn with default props;
     * - `false`: the axis line is not visible;
     * - `object`: passed as props to SVG `<line>` element representing the axis line.
     *
     * @example <YAxis axisLine={false} />
     * @example <YAxis axisLine={{ stroke: 'red', strokeWidth: 2 }} />
     * @defaultValue true
     */
    axisLine?: boolean | SVGProps<SVGLineElement>;
    /**
     * The type of axis.
     *
     * `category`: Treats data as distinct values.
     * Each value is in the same distance from its neighbors, regardless of their actual numeric difference.
     *
     * `number`: Treats data as continuous range.
     * Values that are numerically closer are placed closer together on the axis.
     *
     * `auto`: the type is inferred based on the chart layout.
     *
     * @defaultValue number
     */
    type?: AxisDomainTypeInput;
    /**
     * Specify the domain of axis when the axis is a number axis.
     *
     * If undefined, then the domain is calculated based on the data and dataKeys.
     *
     * The length of domain should be 2, and we will validate the values in domain.
     *
     * Each element in the array can be a number, 'auto', 'dataMin', 'dataMax', a string like 'dataMin - 20', 'dataMax + 100',
     * or a function that accepts a single argument and returns a number.
     *
     * If any element of domain is set to be 'auto', comprehensible scale ticks will be calculated, and the final domain of axis is generated by the ticks.
     * If a function, receives '[dataMin, dataMax]', and must return a computed domain as '[min, max]'.
     *
     * @example <YAxis type="number" domain={['dataMin', 'dataMax']} />
     * @example <YAxis type="number" domain={[0, 'dataMax']} />
     * @example <YAxis type="number" domain={['auto', 'auto']} />
     * @example <YAxis type="number" domain={[0, 'dataMax + 1000']} />
     * @example <YAxis type="number" domain={['dataMin - 100', 'dataMax + 100']} />
     * @example <YAxis type="number" domain={[dataMin => (0 - Math.abs(dataMin)), dataMax => (dataMax * 2)]} />
     * @example <YAxis type="number" domain={([dataMin, dataMax]) => { const absMax = Math.max(Math.abs(dataMin), Math.abs(dataMax)); return [-absMax, absMax]; }} />
     * @example <YAxis type="number" domain={[0, 100]} allowDataOverflow />
     */
    domain?: AxisDomain;
    /**
     * Scale function determines how data values are mapped to visual values.
     * In other words, decided the mapping between data domain and coordinate range.
     *
     * If undefined, or 'auto', the scale function is created internally according to the type of axis and data.
     *
     * You can define a custom scale, either as a string shortcut to a d3 scale, or as a complete scale definition object.
     *
     * @defaultValue auto
     * @example <YAxis scale="log" />
     * @example
     * import { scaleLog } from 'd3-scale';
     * const scale = scaleLog().base(Math.E);
     * <YAxis scale={scale} />
     */
    scale?: ScaleType | CustomScaleDefinition | CustomScaleDefinition<string> | CustomScaleDefinition<number> | CustomScaleDefinition<Date>;
    /**
     * Unique ID that represents this YAxis.
     * Required when there are multiple YAxes.
     *
     * @defaultValue 0
     */
    yAxisId?: string | number;
    /**
     * Defines how the individual label text is rendered.
     * This controls the settings for individual ticks; on a typical axis, there are multiple ticks, depending on your data.
     *
     * If you want to customize the overall axis label, use the `label` prop instead.
     *
     * Options:
     * - `false`: Do not render any tick labels.
     * - `true`: Render tick labels with default settings.
     * - `object`: An object of props to be merged into the internally calculated tick props.
     * - `ReactElement`: A custom React element to be used as the tick label.
     * - `function`: A function that returns a React element for custom rendering of tick labels.
     *
     * @defaultValue true
     */
    tick?: TickProp<YAxisTickContentProps>;
    /**
     * Ticks can be any type when the axis is the type of category
     * Ticks must be numbers when the axis is the type of number
     */
    ticks?: ReadonlyArray<AxisTick>;
    /**
     * Width of the axis in pixels.
     * `auto` will attempt to resize the axis based on its content.
     *
     * @defaultValue 60
     */
    width?: YAxisWidth;
    /**
     * If set true, flips ticks around the axis line, displaying the labels inside the chart instead of outside.
     * @defaultValue false
     */
    mirror?: boolean;
    /**
     * The orientation of axis.
     * @defaultValue left
     */
    orientation?: YAxisOrientation;
    /**
     * Axis padding is the distance between the edge of plot area and the first/last tick.
     *
     * @defaultValue {"top":0,"bottom":0}
     */
    padding?: YAxisPadding;
    /**
     * The minimum gap between two adjacent tick labels
     *
     * @defaultValue 5
     */
    minTickGap?: number;
    /**
     * If set 0, all the ticks will be shown. If set "preserveStart", "preserveEnd" or "preserveStartEnd",
     * the ticks which is to be shown or hidden will be calculated automatically.
     *
     * @defaultValue preserveEnd
     */
    interval?: AxisInterval;
    /**
     * The margin between tick line and tick.
     */
    tickMargin?: number;
}
export type Props = Omit<PresentationAttributesAdaptChildEvent<any, SVGElement>, 'scale' | 'ref'> & YAxisProps;
export declare const yAxisDefaultProps: {
    readonly allowDataOverflow: boolean;
    readonly allowDecimals: boolean;
    readonly allowDuplicatedCategory: boolean;
    readonly angle: number;
    readonly axisLine: true;
    readonly hide: false;
    readonly includeHidden: boolean;
    readonly interval: AxisInterval;
    readonly label: false;
    readonly minTickGap: number;
    readonly mirror: boolean;
    readonly orientation: YAxisOrientation;
    readonly padding: YAxisPadding;
    readonly reversed: boolean;
    readonly scale: ScaleType | CustomScaleDefinition<import("../util/types").CategoricalDomainItem> | CustomScaleDefinition<string> | CustomScaleDefinition<number> | CustomScaleDefinition<Date>;
    readonly tick: TickProp<any>;
    readonly tickCount: number | undefined;
    readonly tickLine: true;
    readonly tickSize: 6;
    readonly type: EvaluatedAxisDomainType;
    readonly width: YAxisWidth;
    readonly yAxisId: 0;
};
/**
 * @consumes CartesianViewBoxContext
 * @provides CartesianLabelContext
 */
export declare const YAxis: ComponentType<Props>;
export {};
